program NEWTONDIFFERENTIATION;
{--------------------------------------------------------------------}
{  Alg6'3.pas   Pascal program for implementing Algorithm 6.3        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 6.3 (Differentiation Based on N+1 Nodes).               }
{  Section   6.2, Numerical Differentiation Formulas, Page 342       }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MaxN = 50;
    FunMax = 6;

  type
    MATRIX = array[0..MaxN, 0..MaxN] of real;
    VECTOR = array[0..MaxN] of real;
    POINTER0 = array[0..MaxN] of integer;
    LETTER = string[8];
    LETTERS = string[200];
    STATUS = (Computing, Done, More, Working);
    DATYPE = (DatPoints, FunPoints);
    ABTYPE = (Given, Equal, Interval, Chebyshev);

  var
    FunType, Inum, M, N, Sub: integer;
    A0, B0, Rnum, T: real;
    D: MATRIX;
    A, AA, C, X, Y: VECTOR;
    Row: POINTER0;
    Resp: CHAR;
    Ans: CHAR;
    Mess: LETTERS;
    Stat, State: STATUS;
    Ytype: DATYPE;
    Xtype: ABTYPE;

  function F (X: real): real;
  begin
    case FunType of
      1:
        F := EXP(X);
      2:
        F := COS(X);
      3: 
        F := SIN(X);
      4: 
        F := SIN(X) / COS(X);
      5: 
        F := ARCTAN(X);
      6: 
        F := LN(X + 1);
    end;
  end;

  procedure PRINTFUN (FunType: integer);
  begin
    case FunType of
      1: 
        WRITE('EXP(X)');
      2: 
        WRITE('COS(X)');
      3: 
        WRITE('SIN(X)');
      4: 
        WRITE('TAN(X)');
      5: 
        WRITE('ARCTAN(X)');
      6: 
        WRITE('LN(X+1)');
    end;
  end;

  procedure DIVIDEDDIFF (X, Y: VECTOR; var D: MATRIX; N: integer);
    var
      J, K: integer;
  begin
    for K := 0 to N do
      D[K, 0] := Y[K];
    for J := 1 to N do
      begin
        for K := J to N do
          D[K, J] := (D[K, J - 1] - D[K - 1, J - 1]) / (X[K] - X[K - J]);
      end;
  end;

  function P (D: MATRIX; X: VECTOR; N: integer; T: real): real;
    var
      K: integer;
      Sum: real;
  begin
    Sum := D[N, N];
    for K := N - 1 downto 0 do
      Sum := Sum * (T - X[K]) + D[K, K];
    P := Sum;
  end;

  procedure DDIFFERENCE (X, Y: VECTOR; var AA: VECTOR; N: integer);
    var
      J, K: integer;
  begin
    for K := 0 to N do
      AA[K] := Y[K];
    for J := 1 to N do
      begin
        for K := N downto J do
          AA[K] := (AA[K] - AA[K - 1]) / (X[K] - X[K - J]);
      end;
  end;

  function DP (AA, X: VECTOR; N: integer): real;
    var
      K: integer;
      Df, Prod, Z: real;
  begin
    Z := X[0];
    Df := AA[1];
    Prod := 1;
    for K := 2 to N do
      begin
        Prod := Prod * (Z - X[K - 1]);
        Df := Df + Prod * AA[K];
      end;
    DP := Df;
  end;

  procedure GETFUNCTION (var FunType: integer);
    var
      K: integer;
  begin
    FunType := 0;
    while FunType = 0 do
      begin
        CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN('     The Newton polynomial will be constructed using the function:');
        WRITELN;
        WRITELN;
        for K := 1 to FunMax do
          begin
            WRITE('     <', K : 2, ' >  F(X) = ');
            PRINTFUN(K);
            WRITELN;
            WRITELN;
          end;
        WRITELN;
        WRITELN;
        Mess := '            SELECT < 1 - 6 > ?  ';
        FunType := 1;
        WRITE(Mess);
        READLN(FunType);
        if FunType < 1 then
          FunType := 1;
        if FunType > 6 then
          FunType := 6;
      end;
  end;

  procedure PRINTPOLY (D: MATRIX; X, Y: VECTOR; N: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    if Ytype = FunPoints then
      begin
        WRITE('F(X) = ');
        PRINTFUN(FunType);
        WRITELN;
        WRITELN;
      end;
    WRITELN('The Newton polynomial of degree ', N : 2, ' is:');
    WRITELN;
    case N of
      1: 
        begin
          WRITELN('P(t) = a  + a [t-x ]');
          WRITELN('        0    1    0 ');
        end;
      2: 
        begin
          WRITELN('P(t) = a  + a [t-x ] + a [t-x ][t-x ]');
          WRITELN('        0    1    0     2    0     1 ');
        end;
      3: 
        begin
          WRITELN('P(t) = a  + a [t-x ] + a [t-x ][t-x ] + a [t-x ][t-x ][t-x ]');
          WRITELN('        0    1    0     2    0     1     3    0     1     2 ');
        end;
      else
        begin
          WRITELN('P(t) = a  + a [t-x ] + a [t-x ][t-x ] + a [t-x ][t-x ][t-x ]');
          WRITELN('        0    1    0     2    0     1     3    0     1     2 ');
          WRITELN;
          WRITELN('      +...+ a [t-x ][t-x ]...[t-x ]');
          WRITELN('             ', N : 1, '    0     1        ', N - 1 : 1);
        end;
    end;
    WRITELN;
    if N < 10 then
      begin
        WRITE('The coefficients:           The abscissas:');
        WRITELN('            The ordinates:');
      end;
    if N > 9 then
      begin
        WRITE('The coefficients:            The abscissas:');
        WRITELN('             The ordinates:');
      end;
    WRITELN;
    for K := 0 to N do
      begin
        if N < 10 then
          begin
            WRITE('A(', K : 1, ')  =', D[K, K] : 14 : 8, '       X(', K : 1, ')  =', X[K] : 12 : 6);
            WRITELN('       Y(', K : 1, ')  =', Y[K] : 12 : 6);
          end;
        if N > 9 then
          begin
            WRITE('A(', K : 2, ')  =', D[K, K] : 14 : 8, '       X(', K : 2, ')  =', X[K] : 12 : 6);
            WRITELN('       Y(', K : 2, ')  =', Y[K] : 12 : 6);
          end;
      end;
  end;

  procedure PRINTXPOLY (A: VECTOR; N: integer);
    var
      K, U, V: integer;
  begin
    CLRSCR;
    WRITELN;
    if Ytype = FunPoints then
      begin
        WRITE('F(X) = ');
        PRINTFUN(FunType);
        WRITELN;
        WRITELN;
      end;
    WRITELN('     When written as an ordinary polynomial,');
    WRITELN('     the polynomial approximation of degree  N = ', N : 2, '  is:');
    WRITELN;
    case N of
      1: 
        begin
          WRITELN('P(x)  =  a  +  a X');
          WRITELN('          0     1');
        end;
      2: 
        begin
          WRITELN('                           2');
          WRITELN('P(x)  =  a   +  a x  +  a X');
          WRITELN('          0      1       2');
        end;
      3: 
        begin
          WRITELN('                           2        3');
          WRITELN('P(x)  =  a   +  a x  +  a x  +  a  X');
          WRITELN('          0      1       2       3');
        end;
      4, 5, 6, 7, 8, 9: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '        ', N : 1);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a x   +  a X');
          WRITELN('          0      1       2            ', N - 1 : 1, '        ', N : 1);
        end;
      10: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '         ', N : 2);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a x   +  a  X');
          WRITELN('          0      1       2           ', N - 1 : 1, '        ', N : 2);
        end;
      else
        begin
          WRITELN('                           2             ', N - 1 : 2, '        ', N : 2);
          WRITELN('P(x)  =  a   +  a x  +  a x   +...+  a  x   +  a  X');
          WRITELN('          0      1       2            ', N - 1 : 2, '        ', N : 2);
        end;
    end;
    WRITELN;
    for K := 0 to TRUNC(N / 2) do                {Print the coefficients}
      begin
        U := 2 * K;                                {of P(X) in two columns}
        V := 2 * K + 1;
        if U <= N then
          begin
            WRITE('A(', U : 1, ') =', A[U] : 15 : 7, '         ');
            if V <= N then
              WRITELN('A(', V : 1, ') =', A[V] : 15 : 7)
            else
              WRITELN;
          end;
      end;
  end;

  procedure REORDER (var X, Y: VECTOR; var Row: POINTER0; var N: integer);
    var
      Cond, J, K, RowK: integer;
      XX, YY: VECTOR;
  begin
    for K := 0 to N do
      begin
        XX[K] := X[K];
        YY[K] := Y[K];
      end;
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('The current order for the abscissas are:');
    WRITELN;
    WRITELN('     k          x ');
    WRITELN('                 k');
    for K := 0 to N do
      WRITELN(K : 6, '      ', X[K] : 15 : 7);
    WRITELN;
    WRITELN;
    case N of
      1: 
        WRITELN('Give a permutation of  0,1  for the new order:');
      2: 
        WRITELN('Give a permutation of  0,1,2  for the new order:');
      else
        WRITELN('Give a permutation of  0,1,...,', N : 0, '  for the new order:');
    end;
    WRITELN;
    Mess := '     k';
    for K := 0 to N do
      begin
        WRITELN;
        Cond := 0;
        while Cond = 0 do
          begin
            Row[K] := 0;
            WRITE(Mess, K : 1, ' = ');
            READLN(Row[K]);
            RowK := Row[K];
            for J := 0 to K - 1 do
              if Row[J] = RowK then
                Cond := 1;
            if (RowK < 0) or (N < RowK) then
              Cond := 1;
            if Cond = 1 then
              begin
                Cond := 0;
                WRITELN;
                WRITELN;
              end
            else
              Cond := 2;
          end;
        WRITELN;
      end;
    WRITELN;
    WRITE('Press the <ENTER> key.  ');
    READLN(ANS);
    WRITELN;
    for K := 0 to N do
      begin
        X[K] := XX[Row[K]];
        Y[K] := YY[Row[K]];
      end;
  end;

  procedure GETPOINTS (var X, Y: VECTOR; var Row: POINTER0; var N: integer; Stat: STATUS);
    type
      CONDTS = (Bad, Enter, Done);
      LETTER = string[1];
    var
      Count, I, J, K, Kbad: integer;
      Xmin, Xmax, Ymin, Ymax: real;
      Valu: real;
      Resp: CHAR;
      Cond: CONDTS;
  begin
    CLRSCR;
    Kbad := -1;
    State := Working;
    if Stat = More then
      begin
        for I := 1 to 6 do
          WRITELN;
        WRITE('Do you want to enter new data points ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Working;
            CLRSCR;
          end;
      end;
    if (Stat = Working) then
      begin
        CLRSCR;
        if Xtype = Given then
          begin
            WRITELN;
            WRITELN('     The Newton polynomial will have degree N,');
            WRITELN;
            WRITELN('     and there must be N+1 points.');
            WRITELN;
            Mess := '     ENTER the degree  N = ';
            N := 1;
            WRITE(Mess);
            READLN(N);
            if N < 1 then
              N := 1;
            if N > 50 then
              N := 50;
          end;
        Kbad := 0;
        for K := 0 to N do
          begin
            if Xtype = Given then
              X[K] := 0;
            Y[K] := 0;
          end;
        CLRSCR;
        WRITELN;
        WRITELN('         Now ENTER the ', (N + 1) : 2, ' points');
        WRITELN;
        WRITELN('     You will have a chance to make changes at the end.');
        WRITELN;
        WRITELN;
        for K := 0 to N do
          begin
            if Xtype = Given then
              begin
                WRITELN;
                Mess := '         x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
              end;
            if Ytype = DatPoints then
              begin
                if Xtype <> Given then
                  begin
                    WRITELN;
                    WRITELN('         x  =', X[K] : 15 : 7);
                    WRITE('          ', K : 0);
                  end;
                Mess := '         y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
                WRITELN;
              end
            else
              begin
                Y[K] := F(X[K]);
              end;
            WRITELN;
          end;
      end;
    Xmin := X[0];
    Ymin := Y[0];
    for K := 1 to N do
      begin
        if (Xmin > X[K]) then
          Xmin := X[K];
        if (Ymin > Y[K]) then
          Ymin := Y[K];
      end;
    Cond := Enter;
    while (Cond = Enter) or (Cond = Bad) do
      begin
        CLRSCR;
        if (Cond = Bad) then
          WRITELN('     The abscissas are NOT distinct.   You MUST change one of them.');
        WRITELN('      k               x                     y');
        WRITELN('                       k                     k');
        WRITELN('----------------------------------------------------------------');
        for K := 0 to N do
          WRITELN('     ', K : 2, '       ', X[K] : 15 : 7, '       ', Y[K] : 15 : 7);
        WRITELN;
        if (Cond <> Bad) then
          begin
            WRITELN;
            if N > 15 then
              begin
                for I := 1 to 9 do
                  WRITELN;
              end;
            WRITE('Are the points o.k. ?  <Y/N>  ');
            READLN(Resp);
            WRITELN;
          end;
        if (Resp = 'N') or (Resp = 'n') or (Cond = Bad) then
          begin
            if N > 14 then
              begin
                WRITELN;
              end;
            WRITELN;
            WRITELN;
            case N of
              1:
                WRITELN('     To change a point select  k = 0,1');
              2: 
                WRITELN('     To change a point select  k = 0,1,2');
              else
                WRITELN('     To change a point select  k = 0,1,...,', N : 2);
            end;
            Mess := '                       ENTER   k = ';
            K := Kbad;
            WRITE(Mess);
            READLN(K);
            if (0 <= K) and (K <= N) then
              begin
                WRITELN;
                if K < 10 then
                  begin
                    WRITELN('     Coordinates of the  current point  (x ,y )  are:');
                    WRITELN('                                          ', K : 1, '  ', k : 1);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '      Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 1, '                              ', K : 1);
                  end
                else
                  begin
                    WRITELN('     Coordinates of the current point  (x  ,y  )  are:');
                    WRITELN('                                         ', K : 2, '  ', k : 2);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '      Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 2, '                             ', K : 2);
                  end;
                Mess := '     NEW   x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
                Mess := '     NEW   y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
              end;
          end
        else
          Cond := Done;
        if (Cond = Bad) then
          Cond := Enter;
        Kbad := -1;
        for J := 0 to N - 1 do
          for K := J + 1 to N do
            if (X[J] = X[K]) then
              begin
                Kbad := K;
                Cond := Bad;
              end;
        Xmax := X[0];
        Xmin := X[0];
        Ymax := Y[0];
        Ymin := Y[0];
        for K := 1 to N do
          begin
            if (Xmax < X[K]) then
              Xmax := X[K];
            if (Xmin > X[K]) then
              Xmin := X[K];
            if (Ymax < Y[K]) then
              Ymax := Y[K];
            if (Ymin > Y[K]) then
              Ymin := Y[K];
          end;
      end;
    WRITELN;
    WRITE('Do you want to reorder the sequence of abscissas ?  <Y/N>  ');
    READLN(Resp);
    WRITELN;
    if (Resp = 'Y') or (Resp = 'y') then
      REORDER(X, Y, Row, N);
  end;

  procedure MESSAGE (var FunType: integer; var Xtype: ABTYPE; var Ytype: DATYPE; var X: VECTOR; var N: integer);
    var
      I, J: integer;
      A0, B0, C0, D0, H: real;
      Ans: CHAR;
  begin
    CLRSCR;
    WRITELN('                    DERIVATIVE OF THE NEWTON POLYNOMIAL');
    WRITELN;
    WRITELN;
    WRITELN('          The Newton polynomial is constructed:');
    WRITELN;
    WRITELN('     P(t) = a  + a [t-x ] + a [t-x ][t-x ] + a [t-x ][t-x ][t-x ]');
    WRITELN('             0    1    0     2    0     1     3    0     1     2 ');
    WRITELN;
    WRITELN('           +...+ a [t-x ][t-x ]...[t-x   ]');
    WRITELN('                  N    0     1        N-1 ');
    WRITELN;
    WRITELN('     based on the N+1 points (x ,y ),(x ,y ),...,(x ,y ).');
    WRITELN('                               0  0    1  1        N  N');
    WRITELN;
    WRITELN('     Then the derivative of P(t) is found at the point (x ,y ),');
    WRITELN('                                                         0  0  ');
    WRITELN('     by using the formula:');
    WRITELN;
    WRITELN('     P`(x ) = a  + a [x - x ] + a [x - x ][x - x ] + a [x - x ][x - x ][x - x ]');
    WRITELN('         0     1    2  0   1     3  0   1   0   2     4  0   1   0   2   0   3 ');
    WRITELN;
    WRITELN('             +...+ a [x - x ][x - x ]...[x - x   ]');
    WRITELN('                    N  0   1   0   2      0   N-1 ');
    WRITELN;
    WRITE('                    Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
    CLRSCR;
    WRITELN;
    WRITELN('     You can choose how to enter the ordinates { y  }.');
    WRITELN('                                                  k   ');
    WRITELN;
    WRITELN(' <1> Enter each value  y  as data.');
    WRITELN('                        k ');
    WRITELN;
    WRITELN(' <2> Use a function to compute  y  =  f(x ) .');
    WRITELN('                                 k       k');
    WRITELN;
    Mess := '     SELECT <1 - 2>  ';
    I := 1;
    WRITE(Mess);
    READLN(I);
    if I <= 1 then
      Ytype := DatPoints
    else
      Ytype := FunPoints;
    CLRSCR;
    WRITELN;
    WRITELN('     You can choose how to enter the abscissas { x  }.');
    WRITELN('                                                  k   ');
    WRITELN;
    WRITELN(' <1> Enter each value  x  as data.');
    WRITELN('                        k ');
    WRITELN;
    WRITELN(' <2> Construct  x  = x  + h  for  k = 1,2,...,N.');
    WRITELN('                 k    0');
    WRITELN;
    WRITELN(' <3> Construct equally spaced points over the interval [A,B],');
    WRITELN;
    WRITELN('     x  = x  + h   for  k = 1,2,...,N      and      h =(B-A)/N.');
    WRITELN('      k    0');
    WRITELN;
    WRITELN(' <4> The abscissas will be the Chebyshev nodes for the interval [A,B].');
    WRITELN;
    WRITELN;
    WRITELN;
    Mess := '     SELECT <1 - 4>  ';
    I := 1;
    WRITE(Mess);
    READLN(I);
    if I < 1 then
      I := 1;
    if I > 4 then
      I := 4;
    if I = 1 then
      Xtype := Given;
    if I = 2 then
      begin
        Xtype := Equal;
        CLRSCR;
        WRITELN;
        WRITELN('You  chose  to construct  x  = x  + h   for  k = 1,2,...,N.');
        WRITELN('                           k    0');
        WRITELN;
        Mess := 'ENTER the starting value  x0 = ';
        X[0] := 0;
        WRITE(Mess);
        READLN(X[0]);
        A0 := X[0];
        WRITELN;
        Mess := '     ENTER the step size  h = ';
        H := 1;
        WRITE(Mess);
        READLN(H);
        WRITELN;
        Mess := 'ENTER the number of steps N = ';
        N := 1;
        WRITE(Mess);
        READLN(N);
        WRITELN;
      end;
    if I = 3 then
      begin
        Xtype := Interval;
        CLRSCR;
        WRITELN;
        WRITELN('You chose to construct equally spaced points over the interval [A,B],');
        WRITELN;
        WRITELN('     x  = x  + h   for  k = 1,2,...,N      and      h =(B-A)/N.');
        WRITELN('      k    0');
        WRITELN;
        Mess := 'ENTER the  left  endpoint  A = ';
        A0 := 0;
        WRITE(Mess);
        READLN(A0);
        WRITELN;
        Mess := 'ENTER the  right endpoint  B = ';
        B0 := 1;
        WRITE(Mess);
        READLN(B0);
        WRITELN;
        Mess := 'ENTER the number of steps  N = ';
        N := 1;
        WRITE(Mess);
        READLN(N);
        WRITELN;
        H := (B0 - A0) / N;
      end;
    if (I = 2) or (I = 3) then
      begin
        for J := 0 to N do
          X[J] := A0 + J * H;
      end;
    if I = 4 then
      begin
        Xtype := Chebyshev;
        CLRSCR;
        WRITELN;
        WRITELN('You chose to use the Chebyshev nodes over the interval [A,B],');
        WRITELN;
        WRITELN('     x  = (A+B)/2 + z (A-B)/2  for  k = 1,2,...,N  where');
        WRITELN('      k              k');
        WRITELN;
        WRITELN('                    z  = cos( (2k+1)Pi/(2N+2) ).');
        WRITELN('                     k');
        WRITELN;
        Mess := 'ENTER the  left  endpoint  A = ';
        A0 := 0;
        WRITE(Mess);
        READLN(A0);
        WRITELN;
        Mess := 'ENTER the  right endpoint  B = ';
        B0 := 1;
        WRITE(Mess);
        READLN(B0);
        WRITELN;
        Mess := 'ENTER the number of steps  N = ';
        N := 1;
        WRITE(Mess);
        READLN(N);
        WRITELN;
        C0 := (A0 + B0) / 2;
        D0 := (A0 - B0) / 2;
        for J := 0 to N do
          begin
            X[J] := C0 + D0 * COS((2 * J + 1) * PI / (2 * N + 2));
          end;
      end;
    if Ytype = FunPoints then
      GETFUNCTION(FunType);
  end;

  procedure MAKEXPOLY (D: MATRIX; X: VECTOR; var C: VECTOR; N: integer);
    var
      J, K: integer;
      Z: real;
  begin
    C[0] := D[N, N];
    for K := 1 to N do
      begin
        Z := X[N - K];
        C[K] := C[K - 1];
        for J := K - 1 downto 1 do
          C[J] := C[J - 1] - Z * C[J];
        C[0] := -Z * C[0] + D[N - K, N - K];
      end;
  end;

  procedure EPOINTS (var A0, B0: real; var M: integer);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    while (Stat = Enter) or (Stat = Change) do
      begin
        if (Stat = Enter) then
          begin
            CLRSCR;
            WRITELN;
            WRITELN;
            WRITELN('                    The minimum abscissa is  ', A0 : 15 : 7);
            WRITELN;
            WRITELN('                    The maximum abscissa is  ', B0 : 15 : 7);
            WRITELN;
            WRITELN;
            Mess := '                 ENTER the left  endpoint  A = ';
            A0 := 0;
            WRITE(Mess);
            READLN(A0);
            Mess := '                 ENTER the right endpoint  B = ';
            B0 := 1;
            WRITE(Mess);
            READLN(B0);
            WRITELN;
            Mess := '                 ENTER number of intervals M = ';
            M := 5;
            WRITE(Mess);
            READLN(M);
            if M < 1 then
              M := 1;
            if M > 100 then
              M := 100;
          end
        else
          begin
            CLRSCR;
            WRITELN;
            WRITELN('                    The left  endpoint is  A =', A0 : 15 : 7);
            WRITELN;
            WRITELN('                    The right endpoint is  B =', B0 : 15 : 7);
            WRITELN;
            WRITELN('                    The number intervals   M = ', M);
            WRITELN;
          end;
        WRITELN;
        WRITE('        Do you want to make a change ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            WRITELN;
            WRITELN;
            WRITELN('     The current left  endpoint is A =', A0 : 15 : 7);
            Mess := '     ENTER  the NEW left  endpoint A = ';
            WRITE(Mess);
            READLN(A0);
            WRITELN;
            WRITELN('     The current right endpoint is B =', B0 : 15 : 7);
            Mess := '     ENTER  the NEW right endpoint B = ';
            WRITE(Mess);
            READLN(B0);
            WRITELN;
            WRITELN('        The current value of M is  M = ', M);
            Mess := '        ENTER  the  NEW  value of  M = ';
            WRITE(Mess);
            READLN(M);
            if (M < 1) then
              M := 1;
            if M > 100 then
              M := 100;
          end
        else
          Stat := Done;
      end;
  end;

  procedure EVALUATE (var AA: VECTOR; X: VECTOR; N: integer);
    var
      Echoice, J: integer;
      H, Valu: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    PRINTPOLY(D, X, Y, N);
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('The value of the derivative at the point (x ,y ) is:');
    WRITELN('                                           0  0     ');
    WRITELN;
    WRITELN(' P`(', X[0] : 15 : 7, '  )  = ', DP(AA, X, N) : 15 : 7);
    WRITELN;
  end;

  procedure DDTABLE (D: MATRIX; X: VECTOR; N: integer);

    var
      J, JB, K, KR: integer;
      H, Valu: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     The divided difference table is:');
    WRITELN;
    WRITELN('      x               F[x ]           F[ , ]          F[ , , ]        F[ , , , ]');
    WRITELN('       k                 k');
    WRITELN;
    if N <= 7 then
      JB := N
    else
      JB := 7;
    for J := 0 to JB do
      begin
        if J <= 3 then
          KR := J
        else
          KR := 3;
        WRITE(X[J] : 13 : 7, '   ');
        for K := 0 to KR - 1 do
          WRITE(D[J, K] : 13 : 7, '   ');
        if KR <= 3 then
          WRITE(D[J, KR] : 13 : 7);
        WRITELN;
      end;
    if N > 4 then
      begin
        WRITELN;
        WRITELN('    x               F[ , , , , ]    F[,,,,,]        F[,,,,,,]       F[,,,,,,,]');
        WRITELN('     k');
        WRITELN;
        for J := 4 to JB do
          begin
            if J <= 7 then
              KR := J
            else
              KR := 7;
            WRITE(X[J] : 13 : 7, '   ');
            for K := 4 to KR - 1 do
              WRITE(D[J, K] : 13 : 7, '   ');
            if KR <= 7 then
              WRITE(D[J, KR] : 13 : 7);
            WRITELN;
          end;
      end;
  end;                                          {End procedure DDTABLE}

begin                                            {Begin Main Program}
  Stat := Working;
  MESSAGE(FunType, Xtype, Ytype, X, N);
  while (Stat = Working) or (Stat = More) do
    begin
      GETPOINTS(X, Y, Row, N, Stat);
      DIVIDEDDIFF(X, Y, D, N);
      DDIFFERENCE(X, Y, AA, N);
      PRINTPOLY(D, X, Y, N);
      WRITELN;
      State := Computing;
      while (State = Computing) do
        begin
          WRITELN;
          WRITELN;
          WRITE('Want to see the  divided  difference table ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'y') or (Ans = 'Y') then
            begin
              DDTABLE(D, X, N);
              WRITELN;
              WRITE('Press the <ENTER> key. ');
              READLN(Ans);
              WRITELN;
            end;
          EVALUATE(AA, X, N);
          WRITELN;
          WRITE('Do you want to see the ordinary polynomial ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'y') or (Ans = 'Y') then
            begin
              MAKEXPOLY(D, X, C, N);
              PRINTXPOLY(C, N);
              WRITELN;
            end;
          WRITELN;
          WRITE('Want to evaluate the same polynomial again ?  <Y/N>  ');
          READLN(Resp);
          WRITELN;
          if (Resp <> 'Y') and (Resp <> 'y') then
            State := Done;
          if (Resp = 'Y') or (Resp = 'y') then
            begin
              CLRSCR;
              Stat := More;
              GETPOINTS(X, Y, Row, N, Stat);
            end;
        end;
      WRITELN;
      WRITE('Want to make a different Newton polynomial ?  <Y/N>  ');
      READLN(Resp);
      WRITELN;
      if (Resp = 'Y') or (Resp = 'y') then
        Stat := More
      else
        Stat := Done;
    end;
end.                                               {End Main Program}

